<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <title>Stripe Payment</title>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body {
            background-color: #f8f9fa;
            padding: 40px 0;
        }
        .payment-container {
            max-width: 650px;
            margin: 0 auto;
        }
        .card {
            border: none;
            border-radius: 10px;
            box-shadow: 0 0 15px rgba(0,0,0,0.1);
        }
        .card-header {
            background-color: #ffffff;
            border-bottom: 1px solid #f0f0f0;
            padding: 20px;
            font-weight: 600;
            font-size: 18px;
            border-radius: 10px 10px 0 0 !important;
        }
        .card-body {
            padding: 30px;
        }
        #card-element {
            padding: 12px 15px;
            border: 1px solid #ced4da;
            border-radius: 5px;
            background-color: white;
            min-height: 45px;
        }
        .btn-primary {
            background-color: #6772e5;
            border-color: #6772e5;
            padding: 12px 16px;
            font-weight: 600;
            width: 100%;
        }
        .btn-primary:hover {
            background-color: #5469d4;
            border-color: #5469d4;
        }
        .spinner-border {
            width: 1rem;
            height: 1rem;
            margin-right: 8px;
        }
        .form-label {
            font-weight: 500;
            margin-bottom: 10px;
        }
    </style>
</head>
<body>
    <div class="container payment-container">
        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-header">
                        Secure Payment with Stripe
                    </div>
                    <div class="card-body">
                        <!-- Alert Messages -->
                        @if (session('success'))
                            <div class="alert alert-success mb-4">
                                {{ session('success') }}
                            </div>
                        @endif
                        
                        @if (session('error'))
                            <div class="alert alert-danger mb-4">
                                {{ session('error') }}
                            </div>
                        @endif

                        <!-- Payment Form -->
                        <form id="payment-form">
                            <!-- Amount Field -->
                            <input type="hidden" id="sale_id" value="{{ $sale_id }}">
                            <input type="hidden" id="customer_id" value="{{ $customer_id }}">

                            <div class="mb-4">
                                <label for="amount" class="form-label">Amount (£)</label>
                                <div class="input-group">
                                    <span class="input-group-text">£</span>
                                    <input type="number" class="form-control" id="amount" name="amount" 
                                           value="{{$amount}}" min="1" step="0.01" required disabled>
                                </div>
                            </div>

                            <!-- Email Field (Optional) -->
                            <div class="mb-4">
                                <label for="email" class="form-label">Email (Optional)</label>
                                <input type="email" class="form-control" id="email" name="email" 
                                       placeholder="email@example.com">
                            </div>

                            <!-- Card Element -->
                            <div class="mb-4">
                                <label for="card-element" class="form-label">Credit or Debit Card</label>
                                <div id="card-element">
                                    <!-- Stripe Card Element will be inserted here -->
                                </div>
                                <!-- Used to display form errors -->
                                <div id="card-errors" role="alert" class="text-danger mt-2 small"></div>
                            </div>

                            <!-- Submit Button -->
                            <div class="mt-4">
                                <button id="submit-button" type="submit" class="btn btn-primary">
                                    <span class="spinner-border spinner-border-sm d-none" id="spinner"></span>
                                    <span id="button-text">Pay Now</span>
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://js.stripe.com/v3/"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Initialize Stripe
            var stripe = Stripe("{{ config('services.stripe.key') }}");
            var elements = stripe.elements();
            const sale_id = document.getElementById('sale_id').value;
            const customer_id = document.getElementById('customer_id').value;
            const amount = document.getElementById('amount').value;

            
            // Style the card Element
            var style = {
                base: {
                    color: '#32325d',
                    fontFamily: 'Arial, sans-serif',
                    fontSmoothing: 'antialiased',
                    fontSize: '16px',
                    '::placeholder': {
                        color: '#aab7c4'
                    }
                },
                invalid: {
                    color: '#fa755a',
                    iconColor: '#fa755a'
                }
            };
            
            // Create and mount the card Element
            var card = elements.create('card', {style: style});
            card.mount('#card-element');
            
            // Handle validation errors
            card.addEventListener('change', function(event) {
                var displayError = document.getElementById('card-errors');
                if (event.error) {
                    displayError.textContent = event.error.message;
                } else {
                    displayError.textContent = '';
                }
            });
            
            // Handle form submission
            var form = document.getElementById('payment-form');
            var submitButton = document.getElementById('submit-button');
            var spinner = document.getElementById('spinner');
            var buttonText = document.getElementById('button-text');
            
            form.addEventListener('submit', function(event) {
                event.preventDefault();
                
                // Disable the submit button and show spinner
                submitButton.disabled = true;
                spinner.classList.remove('d-none');
                buttonText.textContent = 'Processing...';
                
                // Get amount and email
                var amount = document.getElementById('amount').value;
                var email = document.getElementById('email').value;
                
                // Create payment intent on server
                fetch('{{ route('stripe.intent') }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    body: JSON.stringify({
                        amount: amount,
                        email: email,
                        sale_id,
                        customer_id,
                        payment_note: 'Paid via Stripe'
                    })
                })
                .then(function(response) {
                    if (!response.ok) {
                        throw new Error('Network response was not ok');
                    }
                    return response.json();
                })
                .then(function(data) {
                    if (data.error) {
                        displayError(data.error);
                        return;
                    }
                    
                    // Confirm card payment with Stripe
                    return stripe.confirmCardPayment(data.clientSecret, {
                        payment_method: {
                            card: card,
                            billing_details: {
                                email: email || undefined
                            }
                        }
                    });
                })
                .then(function(result) {
                    if (result.error) {
                        // Show error to customer
                        displayError(result.error.message);
                    } else {
                        // Payment succeeded
                        paymentSuccessful(result.paymentIntent.id);
                    }
                })
                .catch(function(error) {
                    displayError('Payment processing failed. Please try again.');
                    console.error('Error:', error);
                });
            });
            
            function displayError(errorMessage) {
                var errorDisplay = document.getElementById('card-errors');
                errorDisplay.textContent = errorMessage;
                submitButton.disabled = false;
                spinner.classList.add('d-none');
                buttonText.textContent = 'Pay Now';
            }
            
            function paymentSuccessful(paymentId) {
                const formContainer = document.querySelector('.card-body');
                
                // Send payment info to backend
                fetch('{{ route('stripe.record') }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    body: JSON.stringify({
                        payment_intent_id: paymentId,
                        amount: amount,
                        sale_id: sale_id,
                        customer_id: customer_id,
                        payment_note: 'Paid via Stripe'
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.status === 'success') {
                        formContainer.innerHTML = `
                            <div class="text-center py-5">
                                <div class="mb-4">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="65" height="65" viewBox="0 0 24 24" fill="none" stroke="#28a745" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"></path>
                                        <polyline points="22 4 12 14.01 9 11.01"></polyline>
                                    </svg>
                                </div>
                                <h4 class="mb-3">Payment Successful!</h4>
                                <p class="text-muted mb-4">Your payment has been processed successfully.</p>
                                <p class="small text-muted mb-4">Payment ID: ${paymentId}</p>
                                <a href="{{url('/')}}" class="btn btn-primary mt-3">Continue Shopping</a>

                            </div>
                        `;
                    } else {
                        displayError('Failed to save payment. Please contact support.');
                    }
                })
                .catch(error => {
                    displayError('Something went wrong. Payment was successful, but could not be saved.');
                    console.error(error);
                });
            }

        });
    </script>
</body>
</html>