<?php

namespace App\Http\Controllers\front;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Stripe\Stripe;
use Stripe\Charge;
use App\User;
use App\Payment;
use App\Account;
use App\Customer;
use App\PaymentWithCreditCard;
use App\Sale;
use Stripe\Exception\ApiErrorException;
use Stripe\PaymentIntent;
use App\PaymentGatewayAccount;

class StripePaymentController extends Controller
{
    /**
     * Display the payment page
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        return view('stripe.payment');
    }

    /**
     * Process the payment
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function processPayment(Request $request)
    {
        // Validate request
        $request->validate([
            'card_no' => 'required',
            'ccExpiryMonth' => 'required',
            'ccExpiryYear' => 'required',
            'cvvNumber' => 'required',
            'amount' => 'required|numeric|min:1',
        ]);

        $input = $request->all();

        // Set your Stripe API key
        Stripe::setApiKey(config('stripe.secret'));

        try {
            // Create a customer
            $customer = Customer::create([
                'email' => $request->email ?? 'customer@example.com',
                'source' => $request->stripeToken,
                'name' => $request->name ?? 'Customer',
            ]);

            // Create a charge
            $charge = Charge::create([
                'customer' => $customer->id,
                'amount' => $input['amount'] * 100, // Amount in cents
                'currency' => 'gbp',
                'description' => 'Payment for order',
            ]);

            // Check if payment was successful
            if ($charge->status == 'succeeded') {
                // You can save transaction details to your database here
                
                // Flash success message
                Session::flash('success', 'Payment successful!');
                return redirect()->back();
            } else {
                Session::flash('error', 'Payment failed!');
                return redirect()->back();
            }
        } catch (ApiErrorException $e) {
            Session::flash('error', $e->getMessage());
            return redirect()->back();
        }
    }

    /**
     * Show payment success page
     *
     * @return \Illuminate\View\View
     */
    public function paymentSuccess()
    {
        return view('stripe.success');
    }

    /**
     * Show payment failure page
     *
     * @return \Illuminate\View\View
     */
    public function paymentFailure()
    {
        return view('stripe.failure');
    }
    
    /**
     * Create payment intent for Stripe Elements
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function createPaymentIntent(Request $request)
    {
        
        Stripe::setApiKey(config('services.stripe.secret'));

        try {
            $paymentIntent = \Stripe\PaymentIntent::create([
                'amount' => $request->amount * 100,
                'currency' => 'gbp',
                'payment_method_types' => ['card'],
            ]);
            
            return response()->json([
                'clientSecret' => $paymentIntent->client_secret
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }


    public function recordPayment(Request $request)
    {
        //dd($request->all());
        $data = $request->all();
        $data['amount'] = $data['amount'] ?? 0.00;

        $sale = Sale::findOrFail($data['sale_id']);
        $customer = Customer::findOrFail($sale->customer_id);

        $sale->paid_amount += $data['amount'];
        $balance = $sale->grand_total - $sale->paid_amount;
        $sale->payment_status = ($balance == 0) ? 4 : 2; // 4 = Paid, 2 = Partial
        $sale->save();

        $user = User::where('is_active',true)->first();
        $account = PaymentGatewayAccount::where('gateway','stripe')->first();
        if($account->gateway == null){
            $account = Account::where('is_active', true)->first();
            $ac_id = $account->id;
        }
        else{
            $ac_id = $account->account_id;
        }

        $payment = new Payment();
        $payment->user_id =  $user->id;;
        $payment->sale_id = $sale->id;
        $payment->account_id = $ac_id ?? null;
        $payment->payment_reference = 'spr-' . date("Ymd") . '-' . date("His");
        $payment->amount = $data['amount'];
        $payment->change = 0;
        $payment->paying_method = 'Credit Card';
        $payment->payment_note = $data['payment_note'] ?? null;
        $payment->save();

        Stripe::setApiKey(config('services.stripe.secret'));

        $intent = PaymentIntent::retrieve($data['payment_intent_id']);
        $customerStripeId = $intent->customer;
        $chargeId = $intent->latest_charge ?? null;

        PaymentWithCreditCard::create([
            'payment_id' => $payment->id,
            'customer_id' => $sale->customer_id,
            'customer_stripe_id' => $customerStripeId,
            'charge_id' => $chargeId,
        ]);

        return response()->json([
            'status' => 'success',
            'message' => 'Payment made successfully.'
        ]);


    }


}